function plot_img(varargin)
%this little function displays a image (jpg, gif, bmp, tiff, etc.) on a
%figure.
%syntax: plot_img('handle',x,'image',y,'x_loc',[x1 x2],'y_loc',[y1 y2],'mod','flipV');
%input: 'handle' = figure handle.  Default: gca
%       'image' = image to be plotted.  Default: open file.
%       'x_loc' and 'y_loc' = vectors, specifies the locations
%       of the pixel centers of image(1,1) and image(M,N).  Image(1,1) is
%       centered over (x_loc(1), y_loc(1)), image(M,N) is centered over
%       (x_loc(end), y_loc(end)), and the pixel centers of the remaining elements
%       of image are spread out evenly between those two points, so that the
%       rectilinear patches are all of equal width and height.  Note: One
%       can enter one point (x_loc,y_loc) and the programe will fill in the
%       end point.
%       'mod' = modifies the image. Parameters are in a cell array:
%           flipV = flip vertically
%           flipH = flip horizontally
%           rot90 = rotate 90deg CCW
%           invert = creates a negative image.
%output: none

%parse the input
[handle,img,x_loc,y_loc,mod] = parse(varargin);

%plot the image
hold(handle,'on');  %hold the current axis for imposition of image

%apply mods to the image
for i = 1:size(mod,2);
    switch mod{1,i}
        case 'flipV'
            img = flipud(img);
        case 'flipH'
            img = fliplr(img);
        case 'rot90'
            img = rot90(img);       %imrotate is also an option, but unnecessary now.
        case 'invert'
            img = uint8(abs(double(img)-255));   %in uint8 grayscale
    end
end

%draw
axes(handle);               %make axis current
image(x_loc,y_loc,img);
hold(handle,'off');


%--------------------------------------------------------------------------
%Subfunction to parse the input
function [handle,img,x_loc,y_loc,mod] = parse(input)
%Initiate Parameter Space
handle = [];
img = [];
x_loc = 1;  %default = upper left corner
y_loc = 1;
mod{1,1} = 'none';   %no mods

%Parse the input
if ~isempty(input)  %there is input
    for i = 1:2:size(input,2);
        if ischar(input{1,i})
            switch input{1,i}
                case 'handle'
                    if isnumeric(input{1,i+1})
                        handle = input{1,i+1};
                    else
                        error(['Error: parameters for handle must follow the form: x']);
                    end
                case 'image'
                    if isnumeric(input{1,i+1})
                        img = input{1,i+1};
                    else
                        error(['Error: parameters for handle must follow the form: y']);
                    end
                case 'x_loc'
                    if isnumeric(input{1,i+1})
                        x_loc = input{1,i+1};
                    else
                        error(['Error: parameters for x_loc must follow the form: [x1 x2]']);
                    end
                case 'y_loc'
                    if isnumeric(input{1,i+1})
                        y_loc = input{1,i+1};
                    else
                        error(['Error: parameters for y_loc must follow the form: [y1 y2]']);
                    end
                case 'mod'
                    if ischar(input{1,i+1})||iscell(input{1,i+1})
                        mod = cellstr(input{1,i+1});    %make sure mod is a cell array
                    else
                        error(['Error: parameters for mod must be: a string or a cell array of strings']);
                    end
                otherwise
                    error(['Error: ',input{1,i},' is not a valid input to plot_img.']);
            end
        else
            error(['Error: You have entered an improper argument. Please check Help for how to use plot_img.']);
        end
    end
end

%make sure all the needed variables are there
if isempty(handle)
    handle = gca;       %default = gca (get current axis)
end
if isempty(img)
    [filename,pathname,filterindex] = uigetfile({'*.tif','TIFF files (*.tif)';...
        '*.jpg','Jpeg files (*.jpg)';'*.gif','GIF files (*.gif)';...
        '*.bmp','Bitmap files (*.bmp)';'*.mat','Mat files (*.mat)';},'Open Stack','Multiselect','off');
    img = imread([pathname,filename]);
end
%not only useful for default, but also when the user only want to enter 1
%point.
if size(y_loc,2)==1 || size(x_loc,2)==1
    [y,x] = size(img);
    y_loc(1,2) = y_loc(1,1)+y-1;
    x_loc(1,2) = x_loc(1,1)+x-1;
end
%we only work in uint8 right now, so all images are converted to uint8
img = im2uint8(img);